-- sql setup script for the IoT Smart Grid demo
--
-- Requires a database with the name PowerConsumption
--
-- Applies To: SQL Server 2016 (or higher); Azure SQL Database
-- Author: Perry Skountrianos (Microsoft)
-- Last Updated: 2016-06-13
use [master]
GO
ALTER DATABASE powerConsumption SET SINGLE_USER WITH ROLLBACK IMMEDIATE;
DROP DATABASE [PowerConsumption]
GO

CREATE DATABASE [PowerConsumption]
 CONTAINMENT = NONE
 ON  PRIMARY 
( NAME = N'PowerConsumption', FILENAME = N'E:\SQLData\PowerConsumption.mdf' , SIZE = 8192KB , FILEGROWTH = 65536KB )
 LOG ON 
( NAME = N'PowerConsumption_log', FILENAME = N'E:\SQLData\PowerConsumption_log.ldf' , SIZE = 8192KB , FILEGROWTH = 65536KB )
GO

USE [PowerConsumption]
GO

SET NOCOUNT ON;
SET XACT_ABORT ON;


-- DROP EXISTING OBJECTS
DROP PROCEDURE IF EXISTS [dbo].[InsertMeterMeasurementHistory] 
GO

DROP PROCEDURE IF EXISTS [dbo].[InsertMeterMeasurement] 
GO
DROP TYPE IF EXISTS [dbo].[udtMeterMeasurement] 
GO
DROP VIEW IF EXISTS [dbo].[vwMeterMeasurement] 
GO
IF OBJECT_ID('dbo.MeterMeasurement', 'U') IS NOT NULL
	ALTER TABLE [dbo].[MeterMeasurement] SET ( SYSTEM_VERSIONING = OFF )
GO
DROP TABLE IF EXISTS [dbo].[MeterMeasurement]
GO
DROP TABLE IF EXISTS [dbo].[MeterMeasurementHistory]
GO

-- CREATE OBJECTS
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE TABLE [dbo].[MeterMeasurementHistory](
	[MeterID] [int] NOT NULL,
	[MeasurementInkWh] [decimal](9, 4) NOT NULL,
	[PostalCode] [nvarchar](10) NOT NULL,
	[MeasurementDate] [datetime2](7) NOT NULL,
	[SysStartTime] [datetime2](7) NOT NULL,
	[SysEndTime] [datetime2](7) NOT NULL
) ON [PRIMARY]

GO

CREATE CLUSTERED COLUMNSTORE INDEX [ix_MeterMeasurementHistory]
    ON [dbo].[MeterMeasurementHistory];


SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE TABLE [dbo].[MeterMeasurement]
(
	[MeterID] [int] NOT NULL,
	[MeasurementInkWh] [decimal](9, 4) NOT NULL,
	[PostalCode] [nvarchar](10) COLLATE SQL_Latin1_General_CP1_CI_AS NOT NULL,
	[MeasurementDate] [datetime2](7) NOT NULL,
	[SysStartTime] [datetime2](7) GENERATED ALWAYS AS ROW START NOT NULL,
	[SysEndTime] [datetime2](7) GENERATED ALWAYS AS ROW END NOT NULL,
	PERIOD FOR SYSTEM_TIME ([SysStartTime], [SysEndTime]),

 PRIMARY KEY NONCLUSTERED 
(
	[MeterID] ASC
)
)WITH (SYSTEM_VERSIONING = ON (HISTORY_TABLE = [dbo].[MeterMeasurementHistory] , DATA_CONSISTENCY_CHECK = ON ) )

GO

CREATE VIEW [dbo].[vwMeterMeasurement]
AS
SELECT	PostalCode,
		DATETIMEFROMPARTS(
			YEAR(MeasurementDate), 
			MONTH(MeasurementDate), 
			DAY(MeasurementDate), 
			DATEPART(HOUR,MeasurementDate), 
			DATEPART(MINUTE,MeasurementDate), 
			DATEPART(ss,MeasurementDate)/1,
			0
		) AS MeasurementDate,
		count(*) AS MeterCount,
		AVG(MeasurementInkWh) AS AvgMeasurementInkWh
FROM	[dbo].[MeterMeasurement] FOR SYSTEM_TIME ALL WITH (NOLOCK)
GROUP BY
		PostalCode,
		DATETIMEFROMPARTS(
		YEAR(MeasurementDate), 
		MONTH(MeasurementDate), 
		DAY(MeasurementDate), 
		DATEPART(HOUR,MeasurementDate), 
		DATEPART(MINUTE,MeasurementDate), 
		DATEPART(ss,MeasurementDate)/1,0)
GO
CREATE TYPE [dbo].[udtMeterMeasurement] AS TABLE(
	[RowID] [int] NOT NULL,
	[MeterID] [int] NOT NULL,
	[MeasurementInkWh] [decimal](9, 4) NOT NULL,
	[PostalCode] [nvarchar](10) COLLATE SQL_Latin1_General_CP1_CI_AS NOT NULL,
	[MeasurementDate] [datetime2](7) NOT NULL)
GO
CREATE PROCEDURE [dbo].[InsertMeterMeasurement] 
	@Batch AS dbo.udtMeterMeasurement READONLY,
	@BatchSize INT
AS
BEGIN 
	DECLARE @i INT = 1
	DECLARE @MeterID INT
	DECLARE @MeasurementInkWh DECIMAL(9, 4)
	DECLARE @PostalCode NVARCHAR(10)
	DECLARE @MeasurementDate DATETIME2(7) 
	
	WHILE (@i <= @BatchSize)
	BEGIN	
	
		SELECT	@MeterID = MeterID,
				@MeasurementInkWh = MeasurementInkWh, 
				@MeasurementDate = MeasurementDate,
				@PostalCode = PostalCode
		FROM	@Batch
		WHERE	RowID = @i
		
		UPDATE	dbo.MeterMeasurement 
		SET		MeasurementInkWh += @MeasurementInkWh,
				MeasurementDate = @MeasurementDate,
				PostalCode = @PostalCode
		WHERE	MeterID = @MeterID							
		
		IF(@@ROWCOUNT = 0)
		BEGIN
			INSERT INTO dbo.MeterMeasurement (MeterID, MeasurementInkWh, PostalCode, MeasurementDate)
			VALUES (@MeterID, @MeasurementInkWh, @PostalCode, @MeasurementDate);			
		END 

		SET @i += 1
	END	
END
GO
